<?php namespace App\Controllers;

use App\Models\PegawaiModel;
use App\Controllers\BaseController;
use CodeIgniter\Exceptions\PageNotFoundException; // Tambahkan ini

class PegawaiAdmin extends BaseController
{
    protected $pegawaiModel;

    public function __construct()
    {
        // Pastikan Anda memanggil helper form
        helper(['form']);
        $this->pegawaiModel = new PegawaiModel();
    }

    // ============================
    // 1. LIST DATA PEGAWAI (admin/news)
    // ============================
    public function index()
    {
        $data = [
            'title'        => 'Daftar Pegawai',
            'pegawai_list' => $this->pegawaiModel->findAll()
        ];
        return view('admin/admin_list_pegawai', $data);
    }

    // ============================
    // 2. HALAMAN CREATE (admin/pegawai/create)
    // ============================
    public function create()
    {
        return view('admin/admin_create_pegawai', [
            'validation' => \Config\Services::validation()
        ]);
    }

    // ============================
    // 3. SIMPAN DATA (CREATE / UPDATE) — FIX UNTUK UPDATE
    // ============================
    public function save()
    {
        // FIX: Ambil status dari tombol 'status'
        $status = $this->request->getPost('status');

        // Validasi
        $rules = [
            'nama_pegawai'  => 'required|min_length[3]',
            'tanggal_lahir' => 'permit_empty|valid_date',
            'jenis_kelamin' => 'required|in_list[Laki-laki,Perempuan]', // Sesuaikan dengan nilai di DB/View
            'foto_pegawai'  => 'max_size[foto_pegawai,1024]|is_image[foto_pegawai]|mime_in[foto_pegawai,image/jpg,image/jpeg,image/png]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('validation', $this->validator);
        }

        // --- Proses Upload Foto ---
        $fileFoto = $this->request->getFile('foto_pegawai');
        $namaFoto = $this->request->getPost('old_foto_pegawai') ?? 'default.jpg'; // Pertahankan foto lama secara default

        if ($fileFoto && $fileFoto->isValid() && !$fileFoto->hasMoved()) {
            $namaFoto = $fileFoto->getRandomName();
            $fileFoto->move('img/pegawai', $namaFoto);
        }
        
        // --- Proses Slug Unik ---
        $baseSlug = url_title($this->request->getPost('nama_pegawai'), '-', true);
        $slug     = $baseSlug;

        $id = $this->request->getPost('id');
        $existing = $this->pegawaiModel->where('slug', $slug)->first();

        // Jika slug sudah ada dan BUKAN data yang sedang diedit, tambahkan timestamp
        if ($existing && (! $id || $existing['id'] != $id)) {
            $slug = $baseSlug . '-' . time();
        }

        // Data untuk disimpan
        $data = [
            'id'            => $id, // ID hanya terisi saat edit
            'nama_pegawai'  => $this->request->getPost('nama_pegawai'),
            'tanggal_lahir' => $this->request->getPost('tanggal_lahir') ?: null,
            'jenis_kelamin' => $this->request->getPost('jenis_kelamin'),
            'author'        => $this->request->getPost('author'),
            'content'       => $this->request->getPost('content'),
            'status'        => $status,
            'slug'          => $slug,
            'foto_pegawai'  => $namaFoto
        ];
        
        // Mode edit (Jika upload foto baru, hapus yang lama)
        if ($id) {
             if ($fileFoto && $fileFoto->isValid() && !$fileFoto->hasMoved()) {
                $old_foto = $this->request->getPost('old_foto_pegawai');
                if ($old_foto && $old_foto !== 'default.jpg' && file_exists('img/pegawai/' . $old_foto)) {
                    @unlink('img/pegawai/' . $old_foto);
                }
            } else {
                // Jika tidak ada upload baru, gunakan foto lama yang sudah ada di $data['foto_pegawai'] dari old_foto_pegawai
            }
        }
        
        // SIMPAN!
        $this->pegawaiModel->save($data);

        $pesan = $status === 'published' ? 'diaktifkan' : 'disimpan sebagai draft';
        return redirect()->to('/admin/news')->with('success', "Pegawai berhasil $pesan!");
    }
    
    // ============================
    // 4. EDIT DATA (admin/pegawai/edit/slug) — FIX 404
    // ============================
    public function edit($slug = null)
    {
        // FIX: Menggunakan where()->first() untuk menghindari asumsi method getPegawai()
        $pegawai = $this->pegawaiModel->where('slug', $slug)->first();

        if (empty($pegawai)) {
            // Jika data tidak ditemukan, lempar 404
            throw PageNotFoundException::forPageNotFound("Pegawai tidak ditemukan.");
        }

        return view('admin/admin_edit_news', [
            'pegawai'    => $pegawai, // FIX: Menggunakan kunci 'pegawai'
            'validation' => \Config\Services::validation()
        ]);
    }

    // ============================
    // 5. DELETE DATA
    // ============================
    public function delete($id)
    {
        $pegawai = $this->pegawaiModel->find($id);

        if ($pegawai && !empty($pegawai['foto_pegawai']) && $pegawai['foto_pegawai'] !== 'default.jpg') {
            $path = 'img/pegawai/' . $pegawai['foto_pegawai'];
            if (file_exists($path)) @unlink($path); // Tambahkan @ untuk menekan error jika gagal hapus
        }

        $this->pegawaiModel->delete($id);
        return redirect()->to('/admin/news')->with('success', 'Pegawai berhasil dihapus.');
    }

    // ============================
    // 6. PREVIEW
    // ============================
    // Catatan: Fungsi ini tidak dipanggil dari rute /admin/news/preview, 
    // tetapi dari rute publik /news/slug.
    // Namun, kita perbaiki kodenya agar konsisten.
    public function preview($slug = null)
    {
        $pegawai = $this->pegawaiModel->where('slug', $slug)->first(); // Gunakan where()

        if (empty($pegawai)) {
            throw PageNotFoundException::forPageNotFound('Pegawai tidak ditemukan.');
        }

        // View ini harusnya news_detail.php
        return view('news_detail', [
            'news' => $pegawai, // Menggunakan kunci 'news' jika news_detail.php menggunakan $news['nama_pegawai']
            'title' => $pegawai['nama_pegawai']
        ]);
    }
}